<?php
/**
 * Products Management
 */

require_once __DIR__ . '/../core/Autoloader.php';

use Core\Auth;
use Core\Security;
use Core\Database;

$auth = new Auth();
$auth->requirePermission('products.view');

$db = Database::getInstance();
$user = $auth->user();

// Get filters
$branchId = $user['branch_id'] ?? $_GET['branch_id'] ?? null;
$search = $_GET['search'] ?? '';

$where = ['1=1'];
$params = [];

if ($branchId && !$auth->hasPermission('*')) {
    $where[] = "p.branch_id = ?";
    $params[] = $branchId;
} elseif ($branchId) {
    $where[] = "p.branch_id = ?";
    $params[] = $branchId;
}

if ($search) {
    $where[] = "(p.name LIKE ? OR p.sku LIKE ? OR p.model LIKE ?)";
    $searchTerm = "%{$search}%";
    $params[] = $searchTerm;
    $params[] = $searchTerm;
    $params[] = $searchTerm;
}

$whereClause = implode(' AND ', $where);

$products = $db->fetchAll(
    "SELECT p.*, b.name as branch_name 
     FROM products p
     LEFT JOIN branches b ON p.branch_id = b.id
     WHERE {$whereClause}
     ORDER BY p.created_at DESC",
    $params
);

$pageTitle = 'إدارة المنتجات';
include '../includes/header.php';
?>

<div class="page-header">
    <h1>إدارة المنتجات</h1>
    <?php if ($auth->hasPermission('products.create')): ?>
        <a href="<?= Core\Helpers::url('products/create.php') ?>" class="btn btn-primary">إضافة منتج جديد</a>
    <?php endif; ?>
</div>

<div class="filters-card">
    <form method="GET" class="filter-form">
        <div class="form-group">
            <input type="text" name="search" placeholder="بحث (اسم، SKU، موديل)" value="<?= Security::sanitizeOutput($search) ?>">
        </div>
        <?php if ($auth->hasPermission('*')): ?>
            <?php
            $branches = $db->fetchAll("SELECT * FROM branches WHERE is_active = 1 ORDER BY name");
            ?>
            <div class="form-group">
                <select name="branch_id">
                    <option value="">جميع الفروع</option>
                    <?php foreach ($branches as $branch): ?>
                        <option value="<?= $branch['id'] ?>" <?= $branchId == $branch['id'] ? 'selected' : '' ?>>
                            <?= Security::sanitizeOutput($branch['name']) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
        <?php endif; ?>
        <button type="submit" class="btn btn-secondary">بحث</button>
        <a href="<?= Core\Helpers::url('products/manage.php') ?>" class="btn btn-secondary">إعادة تعيين</a>
    </form>
</div>

<div class="table-container">
    <table class="data-table">
        <thead>
            <tr>
                <th>SKU</th>
                <th>الاسم</th>
                <th>الموديل</th>
                <th>النوع</th>
                <th>الفرع</th>
                <th>الحالة</th>
                <th>الإجراءات</th>
            </tr>
        </thead>
        <tbody>
            <?php if (empty($products)): ?>
                <tr>
                    <td colspan="7" class="text-center">لا توجد منتجات</td>
                </tr>
            <?php else: ?>
                <?php foreach ($products as $product): ?>
                    <tr>
                        <td><code><?= Security::sanitizeOutput($product['sku']) ?></code></td>
                        <td><?= Security::sanitizeOutput($product['name']) ?></td>
                        <td><?= Security::sanitizeOutput($product['model'] ?? '-') ?></td>
                        <td><?= Security::sanitizeOutput($product['type']) ?></td>
                        <td><?= Security::sanitizeOutput($product['branch_name']) ?></td>
                        <td>
                            <span class="badge <?= $product['is_active'] ? 'badge-success' : 'badge-danger' ?>">
                                <?= $product['is_active'] ? 'نشط' : 'غير نشط' ?>
                            </span>
                        </td>
                        <td>
                            <a href="<?= Core\Helpers::url('products/view.php?id=' . $product['id']) ?>" class="btn btn-sm btn-primary">عرض</a>
                            <?php if ($auth->hasPermission('products.edit')): ?>
                                <a href="<?= Core\Helpers::url('products/edit.php?id=' . $product['id']) ?>" class="btn btn-sm btn-secondary">تعديل</a>
                            <?php endif; ?>
                        </td>
                    </tr>
                <?php endforeach; ?>
            <?php endif; ?>
        </tbody>
    </table>
</div>

<?php include '../includes/footer.php'; ?>
