<?php
/**
 * Create QR Code
 */

require_once __DIR__ . '/../core/Autoloader.php';

use Core\Auth;
use Core\QRCode;
use Core\Security;
use Core\Database;

$auth = new Auth();
$auth->requirePermission('qr.create');

$db = Database::getInstance();
$qrHandler = new QRCode();
$user = $auth->user();

$error = '';
$success = '';

// Get options for dropdowns
$branches = $db->fetchAll("SELECT * FROM branches WHERE is_active = 1 ORDER BY name");
$departments = [];
if ($user['branch_id']) {
    $departments = $db->fetchAll(
        "SELECT * FROM departments WHERE branch_id = ? AND is_active = 1 ORDER BY name",
        [$user['branch_id']]
    );
}
$projects = $db->fetchAll("SELECT * FROM projects WHERE is_active = 1 ORDER BY name");
$products = $db->fetchAll("SELECT * FROM products WHERE is_active = 1 ORDER BY name");
$warehouses = $db->fetchAll("SELECT * FROM warehouses WHERE is_active = 1 ORDER BY name");

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!Security::validateCSRFToken($_POST['_token'] ?? '')) {
        $error = 'Invalid security token';
    } else {
        $data = [
            'branch_id' => $user['branch_id'] ?? $_POST['branch_id'],
            'department_id' => $_POST['department_id'] ?? null,
            'project_id' => $_POST['project_id'] ?? null,
            'product_id' => $_POST['product_id'] ?? null,
            'warehouse_id' => $_POST['warehouse_id'] ?? null,
            'title' => Security::sanitizeInput($_POST['title'] ?? ''),
            'description' => Security::sanitizeInput($_POST['description'] ?? ''),
            'target_type' => $_POST['target_type'] ?? 'custom',
            'target_url' => Security::sanitizeInput($_POST['target_url'] ?? ''),
            'is_active' => isset($_POST['is_active']) ? 1 : 0,
            'password' => $_POST['password'] ?? null,
            'expires_at' => !empty($_POST['expires_at']) ? $_POST['expires_at'] : null,
            'scan_limit' => !empty($_POST['scan_limit']) ? (int)$_POST['scan_limit'] : null,
            'created_by' => $user['id']
        ];
        
            // Auto-generate URL if product or warehouse is selected
        if ($data['product_id']) {
            $data['target_url'] = Core\Helpers::baseUrl() . "/products/view.php?id=" . $data['product_id'];
            $data['target_type'] = 'product';
        } elseif ($data['warehouse_id']) {
            $data['target_url'] = Core\Helpers::baseUrl() . "/warehouses/view.php?id=" . $data['warehouse_id'];
            $data['target_type'] = 'warehouse';
        }
        
        try {
            $qr = $qrHandler->create($data);
            Core\Helpers::redirect('qr/view.php?id=' . $qr['id']);
        } catch (Exception $e) {
            $error = 'Error creating QR code: ' . $e->getMessage();
        }
    }
}

$pageTitle = 'إنشاء رمز QR جديد';
include '../includes/header.php';
?>

<div class="page-header">
    <h1>إنشاء رمز QR جديد</h1>
    <a href="<?= Core\Helpers::url('qr/list.php') ?>" class="btn btn-secondary">رجوع</a>
</div>

<?php if ($error): ?>
    <div class="alert alert-danger"><?= Security::sanitizeOutput($error) ?></div>
<?php endif; ?>

<form method="POST" class="form-card">
    <?= Security::getCSRFTokenField() ?>
    
    <div class="form-grid">
        <div class="form-group">
            <label for="title">العنوان *</label>
            <input type="text" id="title" name="title" required>
        </div>
        
        <div class="form-group">
            <label for="target_type">نوع الهدف *</label>
            <select id="target_type" name="target_type" required>
                <option value="product">منتج</option>
                <option value="warehouse">مخزن</option>
                <option value="invoice">فاتورة</option>
                <option value="warranty">ضمان</option>
                <option value="custom">مخصص</option>
            </select>
        </div>
        
        <div class="form-group">
            <label for="product_id">المنتج</label>
            <select id="product_id" name="product_id">
                <option value="">-- اختر منتج --</option>
                <?php foreach ($products as $product): ?>
                    <option value="<?= $product['id'] ?>"><?= Security::sanitizeOutput($product['name']) ?> (<?= $product['sku'] ?>)</option>
                <?php endforeach; ?>
            </select>
        </div>
        
        <div class="form-group">
            <label for="warehouse_id">المخزن</label>
            <select id="warehouse_id" name="warehouse_id">
                <option value="">-- اختر مخزن --</option>
                <?php foreach ($warehouses as $warehouse): ?>
                    <option value="<?= $warehouse['id'] ?>"><?= Security::sanitizeOutput($warehouse['name']) ?></option>
                <?php endforeach; ?>
            </select>
        </div>
        
        <div class="form-group">
            <label for="department_id">القسم</label>
            <select id="department_id" name="department_id">
                <option value="">-- اختر قسم --</option>
                <?php foreach ($departments as $dept): ?>
                    <option value="<?= $dept['id'] ?>"><?= Security::sanitizeOutput($dept['name']) ?></option>
                <?php endforeach; ?>
            </select>
        </div>
        
        <div class="form-group">
            <label for="project_id">المشروع</label>
            <select id="project_id" name="project_id">
                <option value="">-- اختر مشروع --</option>
                <?php foreach ($projects as $project): ?>
                    <option value="<?= $project['id'] ?>"><?= Security::sanitizeOutput($project['name']) ?></option>
                <?php endforeach; ?>
            </select>
        </div>
        
        <div class="form-group full-width">
            <label for="target_url">رابط الهدف *</label>
            <input type="url" id="target_url" name="target_url" placeholder="https://example.com/page">
            <small>سيتم توليد الرابط تلقائياً عند اختيار منتج أو مخزن. للروابط الداخلية استخدم: <?= Core\Helpers::baseUrl() ?>/product/view.php?id=1</small>
        </div>
        
        <div class="form-group full-width">
            <label for="description">الوصف</label>
            <textarea id="description" name="description" rows="3"></textarea>
        </div>
        
        <div class="form-group">
            <label for="password">كلمة مرور الحماية (اختياري)</label>
            <input type="password" id="password" name="password">
        </div>
        
        <div class="form-group">
            <label for="expires_at">تاريخ الانتهاء</label>
            <input type="datetime-local" id="expires_at" name="expires_at">
        </div>
        
        <div class="form-group">
            <label for="scan_limit">حد المسحات (اختياري)</label>
            <input type="number" id="scan_limit" name="scan_limit" min="1">
        </div>
        
        <div class="form-group">
            <label>
                <input type="checkbox" name="is_active" value="1" checked>
                تفعيل فوري
            </label>
        </div>
    </div>
    
    <div class="form-actions">
        <button type="submit" class="btn btn-primary">إنشاء رمز QR</button>
        <a href="<?= Core\Helpers::url('qr/list.php') ?>" class="btn btn-secondary">إلغاء</a>
    </div>
</form>

<script>
document.getElementById('product_id').addEventListener('change', function() {
    if (this.value) {
        document.getElementById('target_url').value = '<?= Core\Helpers::url("products/view.php?id=") ?>' + this.value;
        document.getElementById('target_type').value = 'product';
    }
});

document.getElementById('warehouse_id').addEventListener('change', function() {
    if (this.value) {
        document.getElementById('target_url').value = '<?= Core\Helpers::url("warehouses/view.php?id=") ?>' + this.value;
        document.getElementById('target_type').value = 'warehouse';
    }
});
</script>

<?php include '../includes/footer.php'; ?>
