<?php
/**
 * Edit QR Code
 */

require_once __DIR__ . '/../core/Autoloader.php';

use Core\Auth;
use Core\QRCode;
use Core\Security;
use Core\Database;

$auth = new Auth();
$auth->requirePermission('qr.edit');

$db = Database::getInstance();
$qrHandler = new QRCode();

$id = $_GET['id'] ?? 0;
$qr = $qrHandler->getById($id);

if (!$qr) {
    Core\Helpers::redirect('404.php');
}

// Check access
if (!$auth->hasPermission('*') && $qr['branch_id'] != $auth->user()['branch_id']) {
    Core\Helpers::redirect('403.php');
}

$user = $auth->user();
$error = '';

// Get options
$departments = [];
if ($user['branch_id']) {
    $departments = $db->fetchAll(
        "SELECT * FROM departments WHERE branch_id = ? AND is_active = 1 ORDER BY name",
        [$user['branch_id']]
    );
}
$projects = $db->fetchAll("SELECT * FROM projects WHERE is_active = 1 ORDER BY name");
$products = $db->fetchAll("SELECT * FROM products WHERE is_active = 1 ORDER BY name");
$warehouses = $db->fetchAll("SELECT * FROM warehouses WHERE is_active = 1 ORDER BY name");

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!Security::validateCSRFToken($_POST['_token'] ?? '')) {
        $error = 'Invalid security token';
    } else {
        $data = [
            'title' => Security::sanitizeInput($_POST['title'] ?? ''),
            'description' => Security::sanitizeInput($_POST['description'] ?? ''),
            'target_type' => $_POST['target_type'] ?? 'custom',
            'target_url' => Security::sanitizeInput($_POST['target_url'] ?? ''),
            'is_active' => isset($_POST['is_active']) ? 1 : 0,
            'password' => $_POST['password'] ?? null,
            'expires_at' => !empty($_POST['expires_at']) ? $_POST['expires_at'] : null,
            'scan_limit' => !empty($_POST['scan_limit']) ? (int)$_POST['scan_limit'] : null
        ];
        
        try {
            $qr = $qrHandler->update($id, $data);
            Core\Helpers::redirect('qr/view.php?id=' . $qr['id']);
        } catch (Exception $e) {
            $error = 'Error updating QR code: ' . $e->getMessage();
        }
    }
}

$pageTitle = 'تعديل رمز QR';
include '../includes/header.php';
?>

<div class="page-header">
    <h1>تعديل رمز QR</h1>
    <a href="<?= Core\Helpers::url('qr/view.php?id=' . $id) ?>" class="btn btn-secondary">رجوع</a>
</div>

<?php if ($error): ?>
    <div class="alert alert-danger"><?= Security::sanitizeOutput($error) ?></div>
<?php endif; ?>

<form method="POST" class="form-card">
    <?= Security::getCSRFTokenField() ?>
    
    <div class="form-grid">
        <div class="form-group">
            <label for="title">العنوان *</label>
            <input type="text" id="title" name="title" value="<?= Security::sanitizeOutput($qr['title']) ?>" required>
        </div>
        
        <div class="form-group">
            <label for="target_type">نوع الهدف *</label>
            <select id="target_type" name="target_type" required>
                <option value="product" <?= $qr['target_type'] == 'product' ? 'selected' : '' ?>>منتج</option>
                <option value="warehouse" <?= $qr['target_type'] == 'warehouse' ? 'selected' : '' ?>>مخزن</option>
                <option value="invoice" <?= $qr['target_type'] == 'invoice' ? 'selected' : '' ?>>فاتورة</option>
                <option value="warranty" <?= $qr['target_type'] == 'warranty' ? 'selected' : '' ?>>ضمان</option>
                <option value="custom" <?= $qr['target_type'] == 'custom' ? 'selected' : '' ?>>مخصص</option>
            </select>
        </div>
        
        <div class="form-group full-width">
            <label for="target_url">رابط الهدف *</label>
            <input type="url" id="target_url" name="target_url" value="<?= Security::sanitizeOutput($qr['target_url']) ?>" required>
        </div>
        
        <div class="form-group full-width">
            <label for="description">الوصف</label>
            <textarea id="description" name="description" rows="3"><?= Security::sanitizeOutput($qr['description']) ?></textarea>
        </div>
        
        <div class="form-group">
            <label for="password">كلمة مرور الحماية (اتركه فارغاً لإبقاء الحالي)</label>
            <input type="password" id="password" name="password">
        </div>
        
        <div class="form-group">
            <label for="expires_at">تاريخ الانتهاء</label>
            <input type="datetime-local" id="expires_at" name="expires_at" 
                   value="<?= $qr['expires_at'] ? date('Y-m-d\TH:i', strtotime($qr['expires_at'])) : '' ?>">
        </div>
        
        <div class="form-group">
            <label for="scan_limit">حد المسحات</label>
            <input type="number" id="scan_limit" name="scan_limit" 
                   value="<?= $qr['scan_limit'] ?>" min="1">
        </div>
        
        <div class="form-group">
            <label>
                <input type="checkbox" name="is_active" value="1" <?= $qr['is_active'] ? 'checked' : '' ?>>
                تفعيل
            </label>
        </div>
    </div>
    
    <div class="form-actions">
        <button type="submit" class="btn btn-primary">حفظ التعديلات</button>
        <a href="<?= Core\Helpers::url('qr/view.php?id=' . $id) ?>" class="btn btn-secondary">إلغاء</a>
    </div>
</form>

<?php include '../includes/footer.php'; ?>
