<?php
/**
 * View Warehouse Details
 */

require_once __DIR__ . '/../core/Autoloader.php';

use Core\Auth;
use Core\Security;
use Core\Database;

$auth = new Auth();
$auth->requireAuth();

$db = Database::getInstance();
$id = $_GET['id'] ?? 0;

$warehouse = $db->fetch(
    "SELECT w.*, b.name as branch_name 
     FROM warehouses w
     LEFT JOIN branches b ON w.branch_id = b.id
     WHERE w.id = ?",
    [$id]
);

if (!$warehouse) {
    Core\Helpers::redirect('404.php');
}

// Get inventory
$inventory = $db->fetchAll(
    "SELECT i.*, p.name as product_name, p.sku, p.type as product_type
     FROM inventory i
     LEFT JOIN products p ON i.product_id = p.id
     WHERE i.warehouse_id = ?
     ORDER BY p.name",
    [$id]
);

// Get related QR codes
$qrCodes = $db->fetchAll(
    "SELECT * FROM qr_codes WHERE warehouse_id = ? ORDER BY created_at DESC",
    [$id]
);

$pageTitle = 'تفاصيل المخزن';
include '../includes/header.php';
?>

<div class="page-header">
    <h1><?= Security::sanitizeOutput($warehouse['name']) ?></h1>
    <?php if ($auth->hasPermission('warehouses.edit')): ?>
        <a href="<?= Core\Helpers::url('warehouses/edit.php?id=' . $warehouse['id']) ?>" class="btn btn-secondary">تعديل</a>
    <?php endif; ?>
</div>

<div class="grid-2">
    <div class="card">
        <h2>معلومات المخزن</h2>
        <table class="info-table">
            <tr>
                <th>الكود:</th>
                <td><code><?= Security::sanitizeOutput($warehouse['code']) ?></code></td>
            </tr>
            <tr>
                <th>الاسم:</th>
                <td><?= Security::sanitizeOutput($warehouse['name']) ?></td>
            </tr>
            <tr>
                <th>الفرع:</th>
                <td><?= Security::sanitizeOutput($warehouse['branch_name']) ?></td>
            </tr>
            <tr>
                <th>الموقع:</th>
                <td><?= Security::sanitizeOutput($warehouse['location'] ?? '-') ?></td>
            </tr>
            <tr>
                <th>المنطقة:</th>
                <td><?= Security::sanitizeOutput($warehouse['area'] ?? '-') ?></td>
            </tr>
            <tr>
                <th>الرف:</th>
                <td><?= Security::sanitizeOutput($warehouse['rack'] ?? '-') ?></td>
            </tr>
            <tr>
                <th>المنطقة:</th>
                <td><?= Security::sanitizeOutput($warehouse['zone'] ?? '-') ?></td>
            </tr>
        </table>
    </div>
    
    <div class="card">
        <h2>إحصائيات المخزون</h2>
        <?php
        $stats = $db->fetch(
            "SELECT 
                COUNT(*) as total_items,
                SUM(quantity) as total_quantity,
                SUM(CASE WHEN status = 'low_stock' THEN 1 ELSE 0 END) as low_stock_count,
                SUM(CASE WHEN status = 'out_of_stock' THEN 1 ELSE 0 END) as out_of_stock_count
             FROM inventory
             WHERE warehouse_id = ?",
            [$id]
        );
        ?>
        <div class="stats-grid" style="grid-template-columns: repeat(2, 1fr);">
            <div class="stat-item">
                <strong><?= $stats['total_items'] ?></strong>
                <span>إجمالي العناصر</span>
            </div>
            <div class="stat-item">
                <strong><?= number_format($stats['total_quantity']) ?></strong>
                <span>إجمالي الكمية</span>
            </div>
            <div class="stat-item">
                <strong style="color: #ffc107;"><?= $stats['low_stock_count'] ?></strong>
                <span>مخزون منخفض</span>
            </div>
            <div class="stat-item">
                <strong style="color: #dc3545;"><?= $stats['out_of_stock_count'] ?></strong>
                <span>نفد المخزون</span>
            </div>
        </div>
    </div>
</div>

<div class="card">
    <h2>المخزون</h2>
    <?php if (empty($inventory)): ?>
        <p>لا يوجد مخزون في هذا الموقع.</p>
    <?php else: ?>
        <div class="table-container">
            <table class="data-table">
                <thead>
                    <tr>
                        <th>SKU</th>
                        <th>اسم المنتج</th>
                        <th>الكمية</th>
                        <th>الحد الأدنى</th>
                        <th>الحالة</th>
                        <th>الإجراءات</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($inventory as $item): ?>
                        <tr>
                            <td><code><?= Security::sanitizeOutput($item['sku']) ?></code></td>
                            <td><?= Security::sanitizeOutput($item['product_name']) ?></td>
                            <td><?= number_format($item['quantity']) ?></td>
                            <td><?= number_format($item['min_threshold']) ?></td>
                            <td>
                                <?php
                                $statusClass = $item['status'] === 'in_stock' ? 'badge-success' : 
                                             ($item['status'] === 'low_stock' ? 'badge-warning' : 'badge-danger');
                                $statusText = $item['status'] === 'in_stock' ? 'متوفر' : 
                                            ($item['status'] === 'low_stock' ? 'منخفض' : 'نفد');
                                ?>
                                <span class="badge <?= $statusClass ?>"><?= $statusText ?></span>
                            </td>
                            <td>
                                <a href="<?= Core\Helpers::url('products/view.php?id=' . $item['product_id']) ?>" class="btn btn-sm btn-primary">عرض المنتج</a>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php endif; ?>
</div>

<?php if (!empty($qrCodes)): ?>
<div class="card">
    <h2>رموز QR المرتبطة</h2>
    <div class="table-container">
        <table class="data-table">
            <thead>
                <tr>
                    <th>الكود</th>
                    <th>العنوان</th>
                    <th>عدد المسحات</th>
                    <th>الإجراءات</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($qrCodes as $qr): ?>
                    <tr>
                        <td><code><?= Security::sanitizeOutput($qr['short_code']) ?></code></td>
                        <td><?= Security::sanitizeOutput($qr['title']) ?></td>
                        <td><?= number_format($qr['scan_count']) ?></td>
                        <td>
                            <a href="<?= Core\Helpers::url('qr/view.php?id=' . $qr['id']) ?>" class="btn btn-sm btn-primary">عرض</a>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</div>
<?php endif; ?>

<?php include '../includes/footer.php'; ?>
